/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArraysClose } from '../test_util';
describeWithFlags('avgPool3d', ALL_ENVS, () => {
    it('x=[2,2,2,1] f=[2,2,2] s=1 p=valid', async () => {
        const x = tf.tensor4d([1, 2, 3, 4, 5, 6, 7, 8], [2, 2, 2, 1]);
        const result = tf.avgPool3d(x, 2, 1, 'valid');
        expect(result.shape).toEqual([1, 1, 1, 1]);
        expectArraysClose(await result.data(), [4.5]);
    });
    it('x=[2,2,2,1] f=[1,2,2] s=1 p=valid', async () => {
        const x = tf.tensor4d([1, 2, 3, 4, 5, 6, 7, 8], [2, 2, 2, 1]);
        const result = tf.avgPool3d(x, [1, 2, 2], 1, 'valid');
        expect(result.shape).toEqual([2, 1, 1, 1]);
        expectArraysClose(await result.data(), [2.5, 6.5]);
    });
    it('x=[1,1,1,1,1] f=[1,1,1] s=1 [0] => [0]', async () => {
        const x = tf.tensor5d([0], [1, 1, 1, 1, 1]);
        const result = tf.avgPool3d(x, 1, 1, 0);
        expect(result.shape).toEqual([1, 1, 1, 1, 1]);
        expectArraysClose(await result.data(), [0]);
    });
    it('x=[1,2,2,2,1] f=[2,2,2] s=1 p=valid', async () => {
        const x = tf.tensor5d([1, 2, 3, 4, 5, 6, 7, 8], [1, 2, 2, 2, 1]);
        const result = tf.avgPool3d(x, 2, 1, 'valid');
        expect(result.shape).toEqual([1, 1, 1, 1, 1]);
        expectArraysClose(await result.data(), [4.5]);
    });
    it('x=[1,2,2,2,1] f=[2,2,2] s=1 p=same', async () => {
        const x = tf.tensor5d([1, 2, 3, 4, 5, 6, 7, 8], [1, 2, 2, 2, 1]);
        const expected = [4.5, 5, 5.5, 6, 6.5, 7, 7.5, 8];
        const result = tf.avgPool3d(x, 2, 1, 'same');
        expect(result.shape).toEqual([1, 2, 2, 2, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('x=[1,3,3,3,1] f=[2,2,2] s=1 p=valid', async () => {
        const x = tf.tensor5d([
            1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
            15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27
        ], [1, 3, 3, 3, 1]);
        const expected = [7.5, 8.5, 10.5, 11.5, 16.5, 17.5, 19.5, 20.5];
        const result = tf.avgPool3d(x, 2, 1, 'valid');
        expect(result.shape).toEqual([1, 2, 2, 2, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('x=[1,3,3,3,1] f=[2,2,2] s=1 p=same', async () => {
        const x = tf.tensor5d([
            1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
            15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27
        ], [1, 3, 3, 3, 1]);
        const expected = [
            7.5, 8.5, 9, 10.5, 11.5, 12, 12, 13, 13.5,
            16.5, 17.5, 18, 19.5, 20.5, 21, 21, 22, 22.5,
            21, 22, 22.5, 24, 25, 25.5, 25.5, 26.5, 27
        ];
        const result = tf.avgPool3d(x, 2, 1, 'same');
        expect(result.shape).toEqual([1, 3, 3, 3, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('x=[1,3,3,3,1] f=[2,2,2] s=1 p=valid, propagates NaNs', async () => {
        const x = tf.tensor5d([
            1, 2, 3, 4, 5, 6, 7, 8, NaN, 10, 11, 12, 13, 14,
            15, 16, 17, 18, 19, 20, 21, NaN, 23, 24, 25, 26, 27
        ], [1, 3, 3, 3, 1]);
        const expected = [7.5, 8.5, 10.5, NaN, NaN, 17.5, NaN, 20.5];
        const result = tf.avgPool3d(x, 2, 1, 'valid');
        expect(result.shape).toEqual([1, 2, 2, 2, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('x=[2,3,3,3,1] f=[2,2,2] s=1 p=valid', async () => {
        const x = tf.tensor5d([
            1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
            15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28,
            29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42,
            43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54
        ], [2, 3, 3, 3, 1]);
        const expected = [
            7.5, 8.5, 10.5, 11.5, 16.5, 17.5, 19.5, 20.5, 34.5, 35.5, 37.5, 38.5,
            43.5, 44.5, 46.5, 47.5
        ];
        const result = tf.avgPool3d(x, 2, 1, 'valid');
        expect(result.shape).toEqual([2, 2, 2, 2, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('x=[1,3,3,3,2] f=[2,2,2] s=1 p=valid', async () => {
        const x = tf.tensor5d([
            1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
            15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28,
            29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42,
            43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54
        ], [1, 3, 3, 3, 2]);
        const expected = [14, 15, 16, 17, 20, 21, 22, 23, 32, 33, 34, 35, 38, 39, 40, 41];
        const result = tf.avgPool3d(x, 2, 1, 'valid');
        expect(result.shape).toEqual([1, 2, 2, 2, 2]);
        expectArraysClose(await result.data(), expected);
    });
    it('x=[1,2,2,2,1] f=[2,2,2] s=1 p=1 roundingMode=floor', async () => {
        const x = tf.tensor5d([1, 2, 3, 4, 5, 6, 7, 8], [1, 2, 2, 2, 1]);
        const expected = [
            1, 1.5, 2, 2, 2.5, 3, 3, 3.5, 4, 3, 3.5, 4, 4, 4.5,
            5, 5, 5.5, 6, 5, 5.5, 6, 6, 6.5, 7, 7, 7.5, 8
        ];
        const result = tf.avgPool3d(x, 2, 1, 1, 'floor');
        expect(result.shape).toEqual([1, 3, 3, 3, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('x=[1,1,1,1,1] f=[1,1,3] s=1 p=valid', async () => {
        // Output tensor would have a dimension of zero, if a certain filter's
        // dimension is larger than the input's.
        const x = tf.tensor5d([1], [1, 1, 1, 1, 1]);
        const expected = [];
        const result = tf.avgPool3d(x, [1, 1, 3], 1, 'valid');
        expect(result.shape).toEqual([1, 1, 1, 0, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('x=[1,1,1,4,1] f=[1,1,1] s=[1,1,2] p=0', async () => {
        // Works if the padding is a number.
        const x = tf.ones([1, 1, 1, 4, 1]);
        const expected = [1, 1];
        const result = tf.avgPool3d(x, [1, 1, 1], [1, 1, 2], 0);
        expect(result.shape).toEqual([1, 1, 1, 2, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('x=[1,1,1,1,1] f=[2,2,2] s=1 p=2', async () => {
        // Works if the padding is larger than filter size.
        const x = tf.ones([1, 1, 1, 1, 1]);
        const expected = [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1,
            1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 1, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
        ];
        const result = tf.avgPool3d(x, [2, 2, 2], 1, 2);
        expect(result.shape).toEqual([1, 4, 4, 4, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('throws when x is not rank 5', async () => {
        // tslint:disable-next-line:no-any
        const x = tf.tensor1d([1]);
        expect(() => tf.avgPool3d(x, 2, 1, 'valid')).toThrowError();
    });
    it('throws when dimRoundingMode is set and pad is same', async () => {
        const x = tf.tensor5d([1], [1, 1, 1, 1, 1]);
        const pad = 'same';
        const dimRoundingMode = 'round';
        expect(() => tf.avgPool3d(x, 2, 1, pad, dimRoundingMode)).toThrowError();
    });
    it('throws when dimRoundingMode is set and pad is valid', async () => {
        const x = tf.tensor5d([1], [1, 1, 1, 1, 1]);
        const pad = 'valid';
        const dimRoundingMode = 'round';
        expect(() => tf.avgPool3d(x, 2, 1, pad, dimRoundingMode)).toThrowError();
    });
    it('throws when dimRoundingMode is set and pad is a non-integer number', async () => {
        const x = tf.tensor5d([1], [1, 1, 1, 1, 1]);
        const pad = 1.2;
        const dimRoundingMode = 'round';
        expect(() => tf.avgPool3d(x, 2, 1, pad, dimRoundingMode)).toThrowError();
    });
    it('throws when passed a non-tensor', () => {
        expect(() => tf.avgPool3d({}, 2, 1, 'valid')).toThrowError();
    });
    it('throws when input dtype is not float32', () => {
        const a = tf.tensor5d([1], [1, 1, 1, 1, 1], 'int32');
        expect(() => tf.avgPool3d(a, 2, 1, 0)).toThrowError();
    });
    it('accepts a tensor-like object', async () => {
        const x = [[[[[0]]]]]; // 1x1x1x1x1
        const result = tf.avgPool3d(x, 1, 1, 0);
        expect(result.shape).toEqual([1, 1, 1, 1, 1]);
        expectArraysClose(await result.data(), [0]);
    });
});
describeWithFlags('avgPool3dBackprop', ALL_ENVS, () => {
    it('gradient x=[2,2,2,1] f=[1,1,1] s=1', async () => {
        const dy = tf.tensor4d([1, 2, 3, 4, 5, 6, 7, 8], [2, 2, 2, 1]);
        const x = tf.ones([2, 2, 2, 1]);
        const expected = [1, 2, 3, 4, 5, 6, 7, 8];
        const dx = tf.grad((x) => tf.avgPool3d(x, 1, 1, 0))(x, dy);
        expect(dx.shape).toEqual(x.shape);
        expectArraysClose(await dx.data(), expected);
    });
    it('gradient x=[1,2,2,2,1] f=[1,1,1] s=1', async () => {
        const dy = tf.tensor5d([1, 2, 3, 4, 5, 6, 7, 8], [1, 2, 2, 2, 1]);
        const x = tf.ones([1, 2, 2, 2, 1]);
        const expected = [1, 2, 3, 4, 5, 6, 7, 8];
        const dx = tf.grad((x) => tf.avgPool3d(x, 1, 1, 0))(x, dy);
        expect(dx.shape).toEqual(x.shape);
        expectArraysClose(await dx.data(), expected);
    });
    it('gradient x=[1,2,2,2,1] f=[2,2,2] s=2', async () => {
        const dy = tf.tensor5d([8], [1, 1, 1, 1, 1]);
        const x = tf.ones([1, 2, 2, 2, 1]);
        const expected = [1, 1, 1, 1, 1, 1, 1, 1];
        const dx = tf.grad((x) => tf.avgPool3d(x, 2, 1, 0))(x, dy);
        expect(dx.shape).toEqual(x.shape);
        expectArraysClose(await dx.data(), expected);
    });
    it('gradient with clone x=[1,2,2,2,1] f=[2,2,2] s=1', async () => {
        const dy = tf.tensor5d([8], [1, 1, 1, 1, 1]);
        const x = tf.ones([1, 2, 2, 2, 1]);
        const expected = [1, 1, 1, 1, 1, 1, 1, 1];
        const dx = tf.grad((x) => tf.avgPool3d(x.clone(), 2, 1, 0).clone())(x, dy);
        expect(dx.shape).toEqual(x.shape);
        expectArraysClose(await dx.data(), expected);
    });
    it('gradient x=[1,3,3,3,1] f=[2,2,2] s=1', async () => {
        const dy = tf.tensor5d([1, 2, 3, 4, 5, 6, 7, 8], [1, 2, 2, 2, 1]);
        const x = tf.ones([1, 3, 3, 3, 1]);
        const expected = [
            0.125, 0.375, 0.25, 0.5, 1.25, 0.75, 0.375, 0.875, 0.5,
            0.75, 1.75, 1, 2, 4.5, 2.5, 1.25, 2.75, 1.5,
            0.625, 1.375, 0.75, 1.5, 3.25, 1.75, 0.875, 1.875, 1
        ];
        const dx = tf.grad((x) => tf.avgPool3d(x, 2, 1, 0))(x, dy);
        expect(dx.shape).toEqual(x.shape);
        expectArraysClose(await dx.data(), expected);
    });
    it('gradient x=[1,4,4,4,1] f=[2,2,2] s=2', async () => {
        const dy = tf.tensor5d([1, 2, 3, 4, 5, 6, 7, 8], [1, 2, 2, 2, 1]);
        const x = tf.ones([1, 4, 4, 4, 1]);
        const expected = [
            0.125, 0.125, 0.25, 0.25, 0.125, 0.125, 0.25, 0.25, 0.375, 0.375,
            0.5, 0.5, 0.375, 0.375, 0.5, 0.5, 0.125, 0.125, 0.25, 0.25,
            0.125, 0.125, 0.25, 0.25, 0.375, 0.375, 0.5, 0.5, 0.375, 0.375,
            0.5, 0.5, 0.625, 0.625, 0.75, 0.75, 0.625, 0.625, 0.75, 0.75,
            0.875, 0.875, 1, 1, 0.875, 0.875, 1, 1, 0.625, 0.625,
            0.75, 0.75, 0.625, 0.625, 0.75, 0.75, 0.875, 0.875, 1, 1,
            0.875, 0.875, 1, 1
        ];
        const dx = tf.grad((x) => tf.avgPool3d(x, 2, 2, 0))(x, dy);
        expect(dx.shape).toEqual(x.shape);
        expectArraysClose(await dx.data(), expected);
    });
    it('gradient x=[1,3,3,3,2] f=[2,2,2] s=1', async () => {
        const dy = tf.tensor5d([1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16], [1, 2, 2, 2, 2]);
        const x = tf.ones([1, 3, 3, 3, 2]);
        const expected = [
            0.125, 0.25, 0.5, 0.75, 0.375, 0.5, 0.75, 1, 2, 2.5, 1.25,
            1.5, 0.625, 0.75, 1.5, 1.75, 0.875, 1, 1.25, 1.5, 3, 3.5,
            1.75, 2, 3.5, 4, 8, 9, 4.5, 5, 2.25, 2.5, 5,
            5.5, 2.75, 3, 1.125, 1.25, 2.5, 2.75, 1.375, 1.5, 2.75, 3,
            6, 6.5, 3.25, 3.5, 1.625, 1.75, 3.5, 3.75, 1.875, 2
        ];
        const dx = tf.grad((x) => tf.avgPool3d(x, 2, 1, 0))(x, dy);
        expect(dx.shape).toEqual(x.shape);
        expectArraysClose(await dx.data(), expected);
    });
    it('gradient x=[2,3,3,3,1] f=[2,2,2] s=1', async () => {
        const dy = tf.tensor5d([1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16], [2, 2, 2, 2, 1]);
        const x = tf.ones([2, 3, 3, 3, 1]);
        const expected = [
            0.125, 0.375, 0.25, 0.5, 1.25, 0.75, 0.375, 0.875, 0.5, 0.75, 1.75,
            1, 2, 4.5, 2.5, 1.25, 2.75, 1.5, 0.625, 1.375, 0.75, 1.5,
            3.25, 1.75, 0.875, 1.875, 1, 1.125, 2.375, 1.25, 2.5, 5.25, 2.75,
            1.375, 2.875, 1.5, 2.75, 5.75, 3, 6, 12.5, 6.5, 3.25, 6.75,
            3.5, 1.625, 3.375, 1.75, 3.5, 7.25, 3.75, 1.875, 3.875, 2
        ];
        const dx = tf.grad((x) => tf.avgPool3d(x, 2, 1, 0))(x, dy);
        expect(dx.shape).toEqual(x.shape);
        expectArraysClose(await dx.data(), expected);
    });
});
//# sourceMappingURL=data:application/json;base64,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